<?php
/**
 * @package     ModScorpionWeather
 * @subpackage  mod_scorpion_weather
 * @copyright   Copyright (C) 2026 Scorpion Computers & Software. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;

/** @var Joomla\CMS\Document\HtmlDocument $document */
$document = Factory::getApplication()->getDocument();

// Load CSS file - use direct path for reliable loading
$cssFile = JPATH_SITE . '/media/mod_scorpion_weather/css/default.min.css';
if (file_exists($cssFile)) {
    $cssPath = Uri::root(true) . '/media/mod_scorpion_weather/css/default.min.css';
    $version = filemtime($cssFile);
    $document->addStyleSheet($cssPath . '?v=' . $version);
}

// Add custom CSS from module parameters
$customCss = trim((string) $params->get('custom_css', ''));
if ($customCss !== '') {
    $document->addStyleDeclaration($customCss);
}

$moduleclass_sfx = $params->get('moduleclass_sfx', '');
$daily           = isset($weather->daily) ? $weather->daily : null;
$days            = ($daily !== null && isset($daily->time)) ? $daily->time : [];
$tempUnit        = $params->get('temperature_unit', 'celsius') === 'fahrenheit' ? '°F' : '°C';
$windUnit        = $params->get('wind_speed_unit', 'kmh');
$windUnitLabel   = ['kmh' => 'km/h', 'ms' => 'm/s', 'mph' => 'mph', 'kn' => 'kn'][$windUnit] ?? 'km/h';

$showPressure      = (bool) $params->get('show_pressure', true);
$showHumidity      = (bool) $params->get('show_humidity', true);
$showVisibility    = (bool) $params->get('show_visibility', true);
$showWind          = (bool) $params->get('show_wind', true);
$showPrecipitation = (bool) $params->get('show_precipitation', true);
$showSunriseSunset = (bool) $params->get('show_sunrise_sunset', true);
$showCredit        = (bool) $params->get('show_credit', true);

$hasData = $weather !== null && !empty($days);
$hourly  = isset($weather->hourly) ? $weather->hourly : null;
$isToday = function($dateStr) {
    $today = new DateTime('today');
    try {
        $date = new DateTime($dateStr);
        return $date->format('Y-m-d') === $today->format('Y-m-d');
    } catch (Exception $e) {
        return false;
    }
};
?>
<div class="mod-scorpion-weather<?php echo $moduleclass_sfx ? ' ' . htmlspecialchars($moduleclass_sfx) : ''; ?>">
    <?php if (!$hasData) : ?>
        <p class="mod-scorpion-weather__unavailable"><?php echo htmlspecialchars(Text::_('MOD_SCORPION_WEATHER_UNAVAILABLE')); ?></p>
        <?php if ($showCredit) : ?>
            <p class="mod-scorpion-weather__credit">
                <a href="https://open-meteo.com/" target="_blank" rel="noopener noreferrer"><?php echo Text::_('MOD_SCORPION_WEATHER_CREDIT'); ?></a>
            </p>
        <?php endif; ?>
</div>
<?php return; endif; ?>
    <div class="mod-scorpion-weather__days" role="list">
        <?php foreach (array_keys($days) as $i) : ?>
            <?php
            $date      = $daily->time[$i] ?? '';
            $dayLabel  = ModScorpionWeatherHelper::formatDayName($date);
            $dateLabel = ModScorpionWeatherHelper::formatDateShort($date);
            $tempMax   = $daily->temperature_2m_max[$i] ?? null;
            $tempMin   = $daily->temperature_2m_min[$i] ?? null;
            $code      = isset($daily->weather_code[$i]) ? (int) $daily->weather_code[$i] : 0;
            $precipSum = $daily->precipitation_sum[$i] ?? null;
            $precipProb = $daily->precipitation_probability_max[$i] ?? null;
            $sunrise   = $showSunriseSunset && !empty($daily->sunrise[$i]) ? $daily->sunrise[$i] : null;
            $sunset    = $showSunriseSunset && !empty($daily->sunset[$i]) ? $daily->sunset[$i] : null;
            $windMax   = $showWind && isset($daily->wind_speed_10m_max[$i]) ? $daily->wind_speed_10m_max[$i] : null;
            $windGust  = $showWind && isset($daily->wind_gusts_10m_max[$i]) ? $daily->wind_gusts_10m_max[$i] : null;
            $iconSlug  = ModScorpionWeatherHelper::getWeatherIconSlug($code);
            
            // Get current hourly values for today only (pressure, humidity, visibility)
            $pressure  = null;
            $humidity  = null;
            $visibility = null;
            if ($i === 0 && $hourly !== null && $isToday($date)) {
                // Find current or nearest hourly data point
                $now = new DateTime();
                $hourlyTimes = $hourly->time ?? [];
                $currentIdx = null;
                foreach ($hourlyTimes as $idx => $hourlyTime) {
                    try {
                        $ht = new DateTime($hourlyTime);
                        if ($ht >= $now || $currentIdx === null) {
                            $currentIdx = $idx;
                            if ($ht >= $now) {
                                break;
                            }
                        }
                    } catch (Exception $e) {
                        continue;
                    }
                }
                if ($currentIdx !== null) {
                    $pressure   = $showPressure && isset($hourly->pressure_msl[$currentIdx]) ? $hourly->pressure_msl[$currentIdx] : null;
                    $humidity   = $showHumidity && isset($hourly->relative_humidity_2m[$currentIdx]) ? $hourly->relative_humidity_2m[$currentIdx] : null;
                    $visibility = $showVisibility && isset($hourly->visibility[$currentIdx]) ? $hourly->visibility[$currentIdx] : null;
                }
            }
            ?>
            <article class="mod-scorpion-weather__day mod-scorpion-weather__day--<?php echo htmlspecialchars($iconSlug); ?>" role="listitem">
                <header class="mod-scorpion-weather__day-header">
                    <time class="mod-scorpion-weather__day-name" datetime="<?php echo htmlspecialchars($date); ?>"><?php echo htmlspecialchars($dayLabel); ?></time>
                    <span class="mod-scorpion-weather__day-date"><?php echo htmlspecialchars($dateLabel); ?></span>
                </header>
                <div class="mod-scorpion-weather__icon" aria-hidden="true">
                    <?php echo ModScorpionWeatherHelper::getWeatherIconSvg($code, 56); ?>
                </div>
                <p class="mod-scorpion-weather__weather-desc" aria-label="<?php echo htmlspecialchars(ModScorpionWeatherHelper::getWeatherLabel($code)); ?>">
                    <?php echo htmlspecialchars(ModScorpionWeatherHelper::getWeatherLabel($code)); ?>
                </p>
                <div class="mod-scorpion-weather__temps">
                    <span class="mod-scorpion-weather__temp mod-scorpion-weather__temp--max" aria-label="Maximum temperature"><?php echo $tempMax !== null ? round($tempMax) . $tempUnit : '—'; ?></span>
                    <span class="mod-scorpion-weather__temp mod-scorpion-weather__temp--sep">/</span>
                    <span class="mod-scorpion-weather__temp mod-scorpion-weather__temp--min" aria-label="Minimum temperature"><?php echo $tempMin !== null ? round($tempMin) . $tempUnit : '—'; ?></span>
                </div>
                <?php if ($showPrecipitation && ($precipSum !== null || $precipProb !== null)) : ?>
                    <div class="mod-scorpion-weather__row">
                        <span class="mod-scorpion-weather__row-label" aria-hidden="true">💧</span>
                        <span class="mod-scorpion-weather__row-value">
                            <?php
                            echo $precipSum !== null && (float) $precipSum > 0 ? round((float) $precipSum, 1) . ' mm' : '0 mm';
                            if ($precipProb !== null) {
                                echo ' <small>(' . round((float) $precipProb) . '%)</small>';
                            }
                            ?>
                        </span>
                    </div>
                <?php endif; ?>
                <?php if ($showWind && ($windMax !== null || $windGust !== null)) : ?>
                    <div class="mod-scorpion-weather__row">
                        <span class="mod-scorpion-weather__row-label" aria-hidden="true">🌬</span>
                        <span class="mod-scorpion-weather__row-value">
                            <?php
                            echo $windMax !== null ? round((float) $windMax) . ' ' . $windUnitLabel : '—';
                            if ($windGust !== null && (float) $windGust > (float) $windMax) {
                                echo ' <small>(' . round((float) $windGust) . ')</small>';
                            }
                            ?>
                        </span>
                    </div>
                <?php endif; ?>
                <?php if ($showSunriseSunset && ($sunrise || $sunset)) : ?>
                    <div class="mod-scorpion-weather__row mod-scorpion-weather__row--sun">
                        <span class="mod-scorpion-weather__row-label" aria-hidden="true">☀</span>
                        <span class="mod-scorpion-weather__row-value">
                            <?php echo ModScorpionWeatherHelper::formatTimeOnly($sunrise); ?> – <?php echo ModScorpionWeatherHelper::formatTimeOnly($sunset); ?>
                        </span>
                    </div>
                <?php endif; ?>
                <?php if ($showPressure && $pressure !== null) : ?>
                    <div class="mod-scorpion-weather__row">
                        <span class="mod-scorpion-weather__row-label" aria-hidden="true">📊</span>
                        <span class="mod-scorpion-weather__row-value"><?php echo round((float) $pressure); ?> hPa</span>
                    </div>
                <?php endif; ?>
                <?php if ($showHumidity && $humidity !== null) : ?>
                    <div class="mod-scorpion-weather__row">
                        <span class="mod-scorpion-weather__row-label" aria-hidden="true">💧</span>
                        <span class="mod-scorpion-weather__row-value"><?php echo round((float) $humidity); ?>%</span>
                    </div>
                <?php endif; ?>
                <?php if ($showVisibility && $visibility !== null) : ?>
                    <div class="mod-scorpion-weather__row">
                        <span class="mod-scorpion-weather__row-label" aria-hidden="true">👁</span>
                        <span class="mod-scorpion-weather__row-value"><?php echo ModScorpionWeatherHelper::formatVisibility((float) $visibility); ?></span>
                    </div>
                <?php endif; ?>
            </article>
        <?php endforeach; ?>
    </div>
    <?php if ($showCredit) : ?>
        <p class="mod-scorpion-weather__credit">
            <a href="https://open-meteo.com/" target="_blank" rel="noopener noreferrer"><?php echo Text::_('MOD_SCORPION_WEATHER_CREDIT'); ?></a>
        </p>
    <?php endif; ?>
</div>
