<?php
/**
 * @package     ModScorpionWeather
 * @subpackage  mod_scorpion_weather
 * @copyright   Copyright (C) 2026 Scorpion Computers & Software. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Date\Date;

/**
 * Helper for mod_scorpion_weather
 *
 * @since  1.0.0
 */
class ModScorpionWeatherHelper
{
    /**
     * Open-Meteo API base URL
     */
    private const API_URL = 'https://api.open-meteo.com/v1/forecast';

    /**
     * Fetch weather forecast from Open-Meteo API with caching.
     *
     * @param   object  $params  Module parameters
     * @return  object|null     Decoded JSON response or null on failure
     * @since   1.0.0
     */
    public static function getWeather(object $params): ?object
    {
        $latitude   = trim((string) $params->get('latitude', '52.3676'));
        $longitude  = trim((string) $params->get('longitude', '4.9041'));
        $cacheMins  = (int) $params->get('cache_minutes', 60);
        $cacheMins  = max(15, min(1440, $cacheMins));

        if ($latitude === '' || $longitude === '') {
            return null;
        }

        $cacheId  = 'mod_scorpion_weather_' . md5($latitude . '|' . $longitude . '|' . $params->get('forecast_days') . '|' . $params->get('cache_minutes') . '|' . $params->get('temperature_unit') . '|' . $params->get('wind_speed_unit') . '|' . $params->get('show_pressure') . '|' . $params->get('show_humidity') . '|' . $params->get('show_visibility') . '|' . $params->get('show_wind') . '|' . $params->get('show_precipitation') . '|' . $params->get('show_sunrise_sunset'));
        $cacheTtl = (int) $cacheMins * 60;
        $cache    = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
            ->createCacheController('output', [
                'defaultgroup' => 'mod_scorpion_weather',
                'lifetime'     => $cacheTtl,
            ]);

        $data = $cache->get($cacheId, 'mod_scorpion_weather');

        if ($data !== false) {
            return $data;
        }

        $data = self::fetchFromApi($params, $latitude, $longitude);

        if ($data !== null) {
            $cache->store($data, $cacheId, 'mod_scorpion_weather');
        }

        return $data;
    }

    /**
     * Perform the actual API request to Open-Meteo.
     *
     * @param   object  $params    Module parameters
     * @param   string  $latitude  Latitude
     * @param   string  $longitude Longitude
     * @return  object|null
     * @since   1.0.0
     */
    private static function fetchFromApi(object $params, string $latitude, string $longitude): ?object
    {
        $forecastDays = (int) $params->get('forecast_days', 7);
        $forecastDays = max(1, min(16, $forecastDays));

        $hourly = [
            'temperature_2m',
            'relative_humidity_2m',
            'precipitation',
            'pressure_msl',
            'visibility',
            'wind_speed_10m',
            'wind_direction_10m',
            'weather_code',
        ];

        $daily = [
            'temperature_2m_max',
            'temperature_2m_min',
            'weather_code',
            'precipitation_sum',
            'precipitation_probability_max',
        ];

        if ((bool) $params->get('show_sunrise_sunset', true)) {
            $daily[] = 'sunrise';
            $daily[] = 'sunset';
        }

        if ((bool) $params->get('show_wind', true)) {
            $daily[] = 'wind_speed_10m_max';
            $daily[] = 'wind_gusts_10m_max';
        }

        // Note: pressure, humidity, and visibility are only available as hourly variables,
        // not as daily aggregates in the Open-Meteo Forecast API
        // These will be shown from hourly data if available, or omitted for daily forecast

        $query = [
            'latitude'             => $latitude,
            'longitude'            => $longitude,
            'forecast_days'        => $forecastDays,
            'timezone'             => 'auto',
            'temperature_unit'     => $params->get('temperature_unit', 'celsius'),
            'wind_speed_unit'      => $params->get('wind_speed_unit', 'kmh'),
            'precipitation_unit'   => 'mm',
        ];

        $url = self::API_URL . '?' . http_build_query($query);

        // Open-Meteo accepts multiple &hourly= and &daily= parameters (avoids 400 on long comma-separated list)
        foreach ($hourly as $h) {
            $url .= '&hourly=' . rawurlencode($h);
        }
        foreach ($daily as $d) {
            $url .= '&daily=' . rawurlencode($d);
        }

        $options = [
            'userAgent' => 'Joomla-ScorpionWeather/1.0',
            'timeout'   => 10,
        ];

        try {
            $response = \Joomla\CMS\Http\HttpFactory::getHttp($options)->get($url);

            if ($response->getStatusCode() !== 200) {
                // Log error for debugging (only in debug mode)
                if (defined('JDEBUG') && JDEBUG) {
                    Factory::getApplication()->enqueueMessage(
                        'Open-Meteo API: HTTP ' . $response->getStatusCode() . ' - ' . substr($response->getBody(), 0, 200),
                        'warning'
                    );
                }
                return null;
            }

            $body = $response->getBody();
            $data = json_decode($body);

            if (json_last_error() !== JSON_ERROR_NONE || !is_object($data)) {
                // Log JSON error for debugging
                if (defined('JDEBUG') && JDEBUG) {
                    Factory::getApplication()->enqueueMessage(
                        'Open-Meteo API: JSON decode error - ' . json_last_error_msg(),
                        'warning'
                    );
                }
                return null;
            }

            if (!empty($data->error) || !isset($data->daily)) {
                // Log API error response
                if (defined('JDEBUG') && JDEBUG) {
                    $errorMsg = isset($data->reason) ? $data->reason : 'Unknown API error';
                    Factory::getApplication()->enqueueMessage(
                        'Open-Meteo API Error: ' . $errorMsg,
                        'warning'
                    );
                }
                return null;
            }

            return $data;
        } catch (Exception $e) {
            // Log exception for debugging
            if (defined('JDEBUG') && JDEBUG) {
                Factory::getApplication()->enqueueMessage(
                    'Open-Meteo API Exception: ' . $e->getMessage(),
                    'error'
                );
            }
            return null;
        }
    }

    /**
     * Get human-readable weather description from WMO code.
     *
     * @param   int  $code  WMO weather code
     * @return  string
     * @since   1.0.0
     */
    public static function getWeatherLabel(int $code): string
    {
        $key = 'MOD_SCORPION_WEATHER_WMO_' . $code;
        $translated = Text::_($key);
        
        // If translation exists (not the key itself), return it
        if ($translated !== $key) {
            return $translated;
        }
        
        // Fallback to English if translation not found
        $labels = [
            0   => 'Clear sky',
            1   => 'Mainly clear',
            2   => 'Partly cloudy',
            3   => 'Overcast',
            45  => 'Fog',
            48  => 'Depositing rime fog',
            51  => 'Light drizzle',
            53  => 'Moderate drizzle',
            55  => 'Dense drizzle',
            56  => 'Light freezing drizzle',
            57  => 'Dense freezing drizzle',
            61  => 'Slight rain',
            63  => 'Moderate rain',
            65  => 'Heavy rain',
            66  => 'Light freezing rain',
            67  => 'Heavy freezing rain',
            71  => 'Slight snow',
            73  => 'Moderate snow',
            75  => 'Heavy snow',
            77  => 'Snow grains',
            80  => 'Slight rain showers',
            81  => 'Moderate rain showers',
            82  => 'Violent rain showers',
            85  => 'Slight snow showers',
            86  => 'Heavy snow showers',
            95  => 'Thunderstorm',
            96  => 'Thunderstorm with slight hail',
            99  => 'Thunderstorm with heavy hail',
        ];

        return $labels[$code] ?? Text::_('MOD_SCORPION_WEATHER_UNKNOWN');
    }

    /**
     * Get icon slug for WMO weather code (for CSS class or SVG use).
     *
     * @param   int  $code  WMO weather code
     * @return  string      One of: clear, partly-cloudy, cloudy, fog, drizzle, rain, snow, thunderstorm
     * @since   1.0.0
     */
    public static function getWeatherIconSlug(int $code): string
    {
        if ($code === 0) {
            return 'clear';
        }
        if ($code >= 1 && $code <= 3) {
            return $code === 1 ? 'partly-cloudy' : ($code === 2 ? 'partly-cloudy' : 'cloudy');
        }
        if ($code >= 45 && $code <= 48) {
            return 'fog';
        }
        if ($code >= 51 && $code <= 67) {
            return $code >= 56 && $code <= 67 ? 'rain' : 'drizzle';
        }
        if ($code >= 71 && $code <= 77) {
            return 'snow';
        }
        if ($code >= 80 && $code <= 82) {
            return 'rain';
        }
        if ($code >= 85 && $code <= 86) {
            return 'snow';
        }
        if ($code >= 95 && $code <= 99) {
            return 'thunderstorm';
        }
        return 'cloudy';
    }

    /**
     * Return inline SVG markup for weather icon (WMO code).
     *
     * @param   int  $code  WMO weather code
     * @param   int  $size  Optional size in pixels (default 48)
     * @return  string     SVG markup
     * @since   1.0.0
     */
    public static function getWeatherIconSvg(int $code, int $size = 48): string
    {
        $slug = self::getWeatherIconSlug($code);
        $s = $size;
        $v = "0 0 {$s} {$s}";
        $c = 'currentColor';
        $svgs = [
            'clear' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--clear" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><circle cx="' . ($s/2) . '" cy="' . ($s/2) . '" r="' . ($s/5) . '" fill="' . $c . '"/><g stroke="' . $c . '" stroke-width="1.5"><line x1="' . ($s/2) . '" y1="4" x2="' . ($s/2) . '" y2="' . ($s/6) . '"/><line x1="' . ($s/2) . '" y1="' . ($s - 4) . '" x2="' . ($s/2) . '" y2="' . ($s - $s/6) . '"/><line x1="4" y1="' . ($s/2) . '" x2="' . ($s/6) . '" y2="' . ($s/2) . '"/><line x1="' . ($s - 4) . '" y1="' . ($s/2) . '" x2="' . ($s - $s/6) . '" y2="' . ($s/2) . '"/><line x1="' . ($s*0.15) . '" y1="' . ($s*0.15) . '" x2="' . ($s*0.28) . '" y2="' . ($s*0.28) . '"/><line x1="' . ($s*0.72) . '" y1="' . ($s*0.72) . '" x2="' . ($s*0.85) . '" y2="' . ($s*0.85) . '"/><line x1="' . ($s*0.72) . '" y1="' . ($s*0.15) . '" x2="' . ($s*0.85) . '" y2="' . ($s*0.28) . '"/><line x1="' . ($s*0.15) . '" y1="' . ($s*0.72) . '" x2="' . ($s*0.28) . '" y2="' . ($s*0.85) . '"/></g></svg>',
            'partly-cloudy' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--partly-cloudy" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><circle cx="' . ($s*0.32) . '" cy="' . ($s*0.32) . '" r="' . ($s/6) . '" fill="' . $c . '"/><ellipse cx="' . ($s*0.58) . '" cy="' . ($s*0.52) . '" rx="' . ($s*0.2) . '" ry="' . ($s*0.12) . '" fill="' . $c . '" opacity="0.9"/><ellipse cx="' . ($s*0.52) . '" cy="' . ($s*0.58) . '" rx="' . ($s*0.22) . '" ry="' . ($s*0.11) . '" fill="' . $c . '" opacity="0.85"/></svg>',
            'cloudy' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--cloudy" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><ellipse cx="' . ($s*0.35) . '" cy="' . ($s*0.45) . '" rx="' . ($s*0.2) . '" ry="' . ($s*0.12) . '" fill="' . $c . '" opacity="0.9"/><ellipse cx="' . ($s*0.65) . '" cy="' . ($s*0.5) . '" rx="' . ($s*0.22) . '" ry="' . ($s*0.14) . '" fill="' . $c . '" opacity="0.85"/><ellipse cx="' . ($s*0.5) . '" cy="' . ($s*0.58) . '" rx="' . ($s*0.25) . '" ry="' . ($s*0.12) . '" fill="' . $c . '" opacity="0.8"/></svg>',
            'fog' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--fog" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><line x1="' . ($s*0.1) . '" y1="' . ($s*0.35) . '" x2="' . ($s*0.9) . '" y2="' . ($s*0.35) . '" stroke="' . $c . '" stroke-width="2" opacity="0.6"/><line x1="' . ($s*0.15) . '" y1="' . ($s*0.5) . '" x2="' . ($s*0.85) . '" y2="' . ($s*0.5) . '" stroke="' . $c . '" stroke-width="2" opacity="0.5"/><line x1="' . ($s*0.05) . '" y1="' . ($s*0.65) . '" x2="' . ($s*0.95) . '" y2="' . ($s*0.65) . '" stroke="' . $c . '" stroke-width="2" opacity="0.6"/></svg>',
            'drizzle' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--rain" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><ellipse cx="' . ($s/2) . '" cy="' . ($s*0.38) . '" rx="' . ($s*0.22) . '" ry="' . ($s*0.12) . '" fill="' . $c . '" opacity="0.9"/><line x1="' . ($s*0.35) . '" y1="' . ($s*0.6) . '" x2="' . ($s*0.4) . '" y2="' . ($s*0.85) . '" stroke="' . $c . '" stroke-width="1.5"/><line x1="' . ($s*0.5) . '" y1="' . ($s*0.55) . '" x2="' . ($s*0.55) . '" y2="' . ($s*0.88) . '" stroke="' . $c . '" stroke-width="1.5"/><line x1="' . ($s*0.65) . '" y1="' . ($s*0.62) . '" x2="' . ($s*0.7) . '" y2="' . ($s*0.9) . '" stroke="' . $c . '" stroke-width="1.5"/></svg>',
            'rain' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--rain" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><ellipse cx="' . ($s/2) . '" cy="' . ($s*0.35) . '" rx="' . ($s*0.24) . '" ry="' . ($s*0.13) . '" fill="' . $c . '" opacity="0.9"/><line x1="' . ($s*0.3) . '" y1="' . ($s*0.58) . '" x2="' . ($s*0.35) . '" y2="' . ($s*0.9) . '" stroke="' . $c . '" stroke-width="2"/><line x1="' . ($s*0.5) . '" y1="' . ($s*0.52) . '" x2="' . ($s*0.55) . '" y2="' . ($s*0.88) . '" stroke="' . $c . '" stroke-width="2"/><line x1="' . ($s*0.7) . '" y1="' . ($s*0.58) . '" x2="' . ($s*0.75) . '" y2="' . ($s*0.92) . '" stroke="' . $c . '" stroke-width="2"/></svg>',
            'snow' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--snow" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><ellipse cx="' . ($s/2) . '" cy="' . ($s*0.38) . '" rx="' . ($s*0.22) . '" ry="' . ($s*0.12) . '" fill="' . $c . '" opacity="0.9"/><path d="M' . ($s/2) . ' ' . ($s*0.5) . ' v' . ($s*0.35) . ' M' . ($s/2) . ' ' . ($s*0.55) . ' l-' . ($s*0.08) . ' ' . ($s*0.08) . ' M' . ($s/2) . ' ' . ($s*0.55) . ' l' . ($s*0.08) . ' ' . ($s*0.08) . ' M' . ($s/2) . ' ' . ($s*0.7) . ' l-' . ($s*0.08) . ' -' . ($s*0.08) . ' M' . ($s/2) . ' ' . ($s*0.7) . ' l' . ($s*0.08) . ' -' . ($s*0.08) . '" stroke="' . $c . '" stroke-width="1.5" fill="none"/></svg>',
            'thunderstorm' => '<svg class="mod-scorpion-weather__icon-svg mod-scorpion-weather__icon-svg--thunderstorm" viewBox="' . $v . '" width="' . $s . '" height="' . $s . '" aria-hidden="true"><ellipse cx="' . ($s/2) . '" cy="' . ($s*0.35) . '" rx="' . ($s*0.24) . '" ry="' . ($s*0.13) . '" fill="' . $c . '" opacity="0.9"/><path d="M' . ($s*0.48) . ' ' . ($s*0.5) . ' L' . ($s*0.38) . ' ' . ($s*0.72) . ' L' . ($s*0.52) . ' ' . ($s*0.72) . ' L' . ($s*0.42) . ' ' . ($s*0.92) . ' L' . ($s*0.58) . ' ' . ($s*0.58) . ' Z" fill="' . $c . '"/></svg>',
        ];
        return $svgs[$slug] ?? $svgs['cloudy'];
    }

    /**
     * Format visibility in meters to readable string (e.g. km).
     *
     * @param   float|null  $meters  Visibility in meters
     * @return  string
     * @since   1.0.0
     */
    public static function formatVisibility(?float $meters): string
    {
        if ($meters === null) {
            return '—';
        }
        if ($meters >= 1000) {
            return round($meters / 1000, 1) . ' km';
        }
        return (int) $meters . ' m';
    }

    /**
     * Format time string (ISO8601) to local time only (e.g. 07:42).
     *
     * @param   string|null  $iso  ISO8601 time string
     * @return  string
     * @since   1.0.0
     */
    public static function formatTimeOnly(?string $iso): string
    {
        if ($iso === null || $iso === '') {
            return '—';
        }
        try {
            $dt = new DateTime($iso);
            return $dt->format('H:i');
        } catch (Exception $e) {
            return '—';
        }
    }

    /**
     * Format date to localized day name (e.g. "Mon" or "Maa").
     *
     * @param   string|null  $iso  ISO8601 date string
     * @return  string
     * @since   1.0.0
     */
    public static function formatDayName(?string $iso): string
    {
        if ($iso === null || $iso === '') {
            return '—';
        }
        try {
            $date = new Date($iso);
            $dayOfWeek = (int) $date->format('w'); // 0 = Sunday, 6 = Saturday
            
            $dayNames = [
                0 => 'SUN',
                1 => 'MON',
                2 => 'TUE',
                3 => 'WED',
                4 => 'THU',
                5 => 'FRI',
                6 => 'SAT',
            ];
            
            $key = 'MOD_SCORPION_WEATHER_DAY_' . $dayNames[$dayOfWeek];
            $translated = Text::_($key);
            
            // If translation exists, return it
            if ($translated !== $key) {
                return $translated;
            }
            
            // Fallback: use English short day name
            $englishDays = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
            return $englishDays[$dayOfWeek];
        } catch (Exception $e) {
            return '—';
        }
    }

    /**
     * Format date to localized date string (e.g. "9 Feb" or "9 feb").
     *
     * @param   string|null  $iso  ISO8601 date string
     * @return  string
     * @since   1.0.0
     */
    public static function formatDateShort(?string $iso): string
    {
        if ($iso === null || $iso === '') {
            return '—';
        }
        try {
            $lang = Factory::getLanguage();
            $date = new Date($iso);
            
            // Get day and month
            $day = (int) $date->format('j');
            $month = (int) $date->format('n'); // 1-12
            
            // Get localized month names
            $monthKeys = [
                1 => 'MOD_SCORPION_WEATHER_MONTH_JANUARY_SHORT',
                2 => 'MOD_SCORPION_WEATHER_MONTH_FEBRUARY_SHORT',
                3 => 'MOD_SCORPION_WEATHER_MONTH_MARCH_SHORT',
                4 => 'MOD_SCORPION_WEATHER_MONTH_APRIL_SHORT',
                5 => 'MOD_SCORPION_WEATHER_MONTH_MAY_SHORT',
                6 => 'MOD_SCORPION_WEATHER_MONTH_JUNE_SHORT',
                7 => 'MOD_SCORPION_WEATHER_MONTH_JULY_SHORT',
                8 => 'MOD_SCORPION_WEATHER_MONTH_AUGUST_SHORT',
                9 => 'MOD_SCORPION_WEATHER_MONTH_SEPTEMBER_SHORT',
                10 => 'MOD_SCORPION_WEATHER_MONTH_OCTOBER_SHORT',
                11 => 'MOD_SCORPION_WEATHER_MONTH_NOVEMBER_SHORT',
                12 => 'MOD_SCORPION_WEATHER_MONTH_DECEMBER_SHORT',
            ];
            
            $monthKey = $monthKeys[$month] ?? 'MOD_SCORPION_WEATHER_MONTH_JANUARY_SHORT';
            $monthName = Text::_($monthKey);
            
            // If translation doesn't exist, use English fallback
            if ($monthName === $monthKey) {
                $englishMonths = ['', 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
                $monthName = $englishMonths[$month];
            }
            
            return $day . ' ' . $monthName;
        } catch (Exception $e) {
            // Fallback to simple format
            try {
                $dt = new DateTime($iso);
                return $dt->format('j M');
            } catch (Exception $e2) {
                return '—';
            }
        }
    }
}
