<?php
/**
 * @package     Scorpion Opening Hours
 * @subpackage  com_scorpionopeninghours
 * @author      Scorpion Computers and Software
 * @copyright   Copyright (C) Scorpion Computers and Software. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Component\ComponentHelper;

/**
 * Scorpion Opening Hours Helper Class
 */
class ScorpionopeninghoursHelper 
{
    // Setting up private variables to use
    private $componentName = 'com_scorpionopeninghours';
    private $componentNameFull = 'Scorpion Opening Hours';
    private $currentUser;
    private $openingHoursData;
    private $document;
    private $baseURL;
    private $moduleID;
    private $moreStyle;
    private $chosenTemplate;
    private $openingHoursList;
    private $selectedTimezone;
    private $showFutureHours;
    private $numberFutureHours;

    /**
     * Constructor
     *
     * @param   mixed   $openingHoursData  Opening hours data (object or ID)
     * @param   string  $moduleId          Module ID for styling
     * @param   string  $template          Template name
     * @param   string  $moreStyle         Additional CSS styling from module
     * @param   int     $showFutureHours   Whether to show future additional hours (from module)
     * @param   int     $numberFutureHours Number of future hours to show (from module)
     */
    public function __construct($openingHoursData = null, $moduleId = 'default', $template = 'default', $moreStyle = '', $showFutureHours = 1, $numberFutureHours = 3) 
    {
        if (is_numeric($openingHoursData)) {
            // Load opening hours by ID
            $this->openingHoursData = $this->getOpeningHoursById($openingHoursData);
        } elseif (is_object($openingHoursData)) {
            // Use provided object
            $this->openingHoursData = $openingHoursData;
        } else {
            // Load default opening hours
            $this->openingHoursData = $this->getDefaultOpeningHours();
        }

        if (!$this->openingHoursData) {
            throw new Exception('No opening hours data found');
        }

        $this->moduleID = $moduleId;
        $this->moreStyle = $moreStyle; // CSS styling from module parameters
        $this->chosenTemplate = $template;
        $this->showFutureHours = $showFutureHours; // Whether to show future hours (from module)
        $this->numberFutureHours = $numberFutureHours; // Number of future hours to show (from module)
        
        // Build opening hours list array
        $this->openingHoursList = array(
            $this->openingHoursData->monday_times ?? '08:30 - 18:00',
            $this->openingHoursData->tuesday_times ?? '08:30 - 18:00',
            $this->openingHoursData->wednesday_times ?? '08:30 - 18:00',
            $this->openingHoursData->thursday_times ?? '08:30 - 18:00',
            $this->openingHoursData->friday_times ?? '08:30 - 18:00',
            $this->openingHoursData->saturday_times ?? '08:30 - 18:00',
            $this->openingHoursData->sunday_times ?? 'CLOSED'
        );
        
        $this->selectedTimezone = $this->getUsersOrSelectedTimezone();
        
        // Setting default variables to use in the component
        $this->moduleID = 'scs-opening-hours-' . $moduleId;
        $this->document = Factory::getDocument();
        $this->baseURL = Uri::root(true);
    }

    /**
     * Get opening hours by ID
     *
     * @param   int  $id  Opening hours ID
     *
     * @return  object|null
     */
    protected function getOpeningHoursById($id)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__scorpion_opening_hours')
            ->where('id = ' . (int) $id)
            ->where('published = 1');
        
        $db->setQuery($query);
        return $db->loadObject();
    }

    /**
     * Get default opening hours (first published record)
     *
     * @return  object|null
     */
    protected function getDefaultOpeningHours()
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__scorpion_opening_hours')
            ->where('published = 1')
            ->order('ordering ASC, id ASC');
        
        $db->setQuery($query, 0, 1);
        return $db->loadObject();
    }

    /**
     * Get all published opening hours
     *
     * @return  array
     */
    public static function getAllOpeningHours()
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__scorpion_opening_hours')
            ->where('published = 1')
            ->order('ordering ASC, title ASC');
        
        $db->setQuery($query);
        return $db->loadObjectList();
    }

    /**
     * Get field value with fallback
     *
     * @param   string  $fieldName     Field name
     * @param   mixed   $defaultValue  Default value
     *
     * @return  mixed
     */
    public function getFieldValue($fieldName, $defaultValue) 
    {
        return $this->openingHoursData->$fieldName ?? $defaultValue;
    }

    /**
     * Add the basic CSS file from this component
     */
    public function addBasicCSS() 
    {
        $urlCSS = $this->baseURL . '/components/com_scorpionopeninghours/assets/css/scorpionopeninghours.css';
        $this->document->addStyleSheet($urlCSS);
        
        // Add the specific CSS file for this template
        $templateCSS = $this->baseURL . '/components/com_scorpionopeninghours/tmpl/' . $this->chosenTemplate . '.css';
        if (file_exists(JPATH_SITE . '/components/com_scorpionopeninghours/tmpl/' . $this->chosenTemplate . '.css')) {
            $this->document->addStylesheet($templateCSS);
        }
    }

    /**
     * Add extra CSS styling
     *
     * @param   string  $extraStyle  Additional CSS
     */
    public function addExtraCSS(&$extraStyle) 
    {
        // Only add CSS if there's actually some CSS to add
        $hasExtraStyle = !empty(trim($extraStyle));
        $hasMoreStyle = !empty(trim($this->moreStyle));
        
        if ($hasExtraStyle || $hasMoreStyle) {
            // Adding style tag to the top of the HTML document for better support by browsers
            $stylelink = '<style>' . "\r\n";
            $stylelink .= '/* Styling for ' . $this->componentNameFull . ' */' . "\r\n";
            
            if ($hasExtraStyle) {
                $stylelink .= $extraStyle . "\r\n";
            }
            
            if ($hasMoreStyle) {
                $stylelink .= $this->moreStyle . "\r\n";
            }
            
            $stylelink .= '</style>';
            $this->document->addCustomTag($stylelink);
        }
    }

    /**
     * Add basic HTML wrapper
     *
     * @param   string  $StartOrEnd    'start' or 'end'
     * @param   string  $templateName  Template name
     */
    public function addBasicHTML($StartOrEnd, $templateName = null) 
    {
        $htmlCode = '';
        if ($StartOrEnd == 'start') {
            $htmlCode .= "\r\n" . '<!-- ' . $this->componentNameFull . ' - BEGIN -->' . "\r\n";
            $htmlCode .= '<div id="' . $this->moduleID . '" class="scs-opening-hours" data-template="' . $templateName . '">' . "\r\n";
        } elseif ($StartOrEnd == 'end') {
            $htmlCode .= '</div>' . "\r\n";
            $htmlCode .= '<!-- ' . $this->componentNameFull . ' - END -->' . "\r\n";
        }
        echo $htmlCode;
    }

    /**
     * Get additional text HTML
     *
     * @param   string  $BeforeOrAfter  'before' or 'after'
     *
     * @return  string
     */
    public function getAdditionalTextHTML($BeforeOrAfter) 
    {
        $htmlCode = '';
        if ($BeforeOrAfter == 'before') {
            if ($this->openingHoursData->use_text_before ?? false) {
                $htmlCode .= '' . "\r\n";
                $htmlCode .= '<div>' . "\r\n";
                $htmlCode .= ($this->openingHoursData->text_before ?? '') . "\r\n";
                $htmlCode .= '</div>' . "\r\n";
            }
        } elseif ($BeforeOrAfter == 'after') {
            if ($this->openingHoursData->use_text_after ?? false) {
                $htmlCode .= '' . "\r\n";
                $htmlCode .= '<div>' . "\r\n";
                $htmlCode .= ($this->openingHoursData->text_after ?? '') . "\r\n";
                $htmlCode .= '</div>' . "\r\n";
            }
        }
        return $htmlCode;
    }

    /**
     * Get user's timezone or selected timezone
     *
     * @return  string
     */
    public function getUsersOrSelectedTimezone() 
    {
        // See if a user is logged in
        $this->currentUser = Factory::getApplication()->getIdentity();
        if ($this->currentUser->id <> 0) {
            // If a user is logged in get it's selected timezone
            $UsersTimezone = $this->currentUser->getParam('timezone');
            if ($UsersTimezone) {
                // If the user has set a specific timezone, we use that instead of the timezone selected in the component
                return $UsersTimezone;
            } else {
                return $this->openingHoursData->timezone ?? 'Europe/Amsterdam';
            }
        } else {
            return $this->openingHoursData->timezone ?? 'Europe/Amsterdam';
        }
    }
    
    /**
     * Get current day opening times
     *
     * @param   int  $dayNumber  Day number (1-7)
     *
     * @return  string
     */
    public function getCurrentDayOpeningtimes($dayNumber) 
    {
        return $this->openingHoursList[$dayNumber - 1];
    }

    /**
     * Check if given day number is current day
     *
     * @param   int  $dayNumber  Day number (1-7)
     *
     * @return  boolean
     */
    public function isThisCurrentDay($dayNumber) 
    {
        $currentDayNr = Factory::getDate()->format('N');
        return ($dayNumber == $currentDayNr);
    }

    /**
     * Get current day CSS class
     *
     * @param   int  $dayNumber  Day number (1-7)
     *
     * @return  string
     */
    public function getCurrentDayClass($dayNumber) 
    {
        $result = '';
        $currentDayNr = Factory::getDate()->format('N');
        if ($dayNumber == $currentDayNr) {
            if ($this->openingHoursData->highlight_today ?? true) {
                $result .= ' scs-current-day';
            }
            if ($this->areWeOpenNow()) {
                $result .= ' scs-open';
            } else {
                $result .= ' scs-closed';
            }
        }
        return $result;
    }

    /**
     * Check if we are currently open
     *
     * @return  boolean
     */
    public function areWeOpenNow() 
    {
        $userTimezone = $this->getUsersOrSelectedTimezone();
        $currentTime = new DateTime('now', new DateTimeZone($userTimezone));
        $currentDate = clone $currentTime;
        $currentDate->setTime(0, 0, 0);
        
        // First check if there are any additional opening hours for today
        $additionalHours = $this->getAdditionalOpeningHours();
        foreach ($additionalHours as $hour) {
            $dateFrom = clone $hour['date_from'];
            $dateTo = clone $hour['date_to'];
            $dateFrom->setTime(0, 0, 0);
            $dateTo->setTime(23, 59, 59);
            
            // Check if today falls within the additional opening hours date range
            if ($currentDate >= $dateFrom && $currentDate <= $dateTo) {
                if ($hour['open_or_closed'] === 'closed') {
                    // Additional hours say we're closed today
                    return false;
                } else {
                    // Additional hours say we're open today
                    if ($hour['fullday'] === '1') {
                        // Full day open
                        return true;
                    } else {
                        // Check specific times
                        $startTime = $hour['start_time'];
                        $endTime = $hour['end_time'];
                        $currentTimeString = $currentTime->format('H:i');
                        
                        if (!empty($startTime) && !empty($endTime)) {
                            if (strtotime($currentTimeString) >= strtotime($startTime) && strtotime($currentTimeString) <= strtotime($endTime)) {
                                return true;
                            } else {
                                return false;
                            }
                        }
                    }
                }
            }
        }
        
        // No additional hours for today, check regular opening hours
        $result = false;
        $currentDayNr = Factory::getDate()->format('N');
        // retrieve the opening hours of today
        $currentOpeningTime = str_replace(' ', '', $this->openingHoursList[$currentDayNr - 1]);
        
        // Check if it's marked as closed
        if (strtoupper($currentOpeningTime) === 'CLOSED' || strtoupper($currentOpeningTime) === 'GESLOTEN') {
            return false;
        }
        
        // and split them into usable timeslots
        $times = explode('-', $currentOpeningTime, 99);
        if (count($times) >= 2) {
            // only get the time to compare without the date, otherwise the comparison will go wrong
            $currentTimeString = $currentTime->format('H:i');
            if (strtotime($currentTimeString) >= strtotime($times[0]) && strtotime($currentTimeString) <= strtotime($times[1])) {
                $result = true;
            }
        }
        
        return $result;
    }

    /**
     * Get open/closed text
     *
     * @return  string
     */
    public function getOpenClosedText() 
    {
        $result = '';
        if (strlen($this->openingHoursData->pre_open_closed_text ?? '') > 0) {
            $result .= '<span class="scs-pre-open-closed">';
            $result .= ($this->openingHoursData->pre_open_closed_text ?? '') . '&nbsp;';
            $result .= '</span>';
        }
        if ($this->areWeOpenNow()) {
            $result .= '<span class="scs-open-closed scs-open">';
            $result .= $this->openingHoursData->open_text ?? 'OPEN';
        } else {
            $result .= '<span class="scs-open-closed scs-closed">';
            $result .= $this->openingHoursData->closed_text ?? 'CLOSED';
        }
        $result .= '</span>';
        return $result;
    }

    /**
     * Get open/closed CSS class
     *
     * @return  string
     */
    public function getOpenClosedClass() 
    {
        $result = '';
        if ($this->areWeOpenNow()) {
            $result .= 'scs-open';
        } else {
            $result .= 'scs-closed';
        }
        return $result;
    }

    /**
     * Get additional opening hours (now global, not tied to specific opening hours)
     *
     * @return  array
     */
    public function getAdditionalOpeningHours() 
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__scorpion_opening_hours_additional')
            ->where('published = 1')
            ->order('date_from ASC');

        $db->setQuery($query);
        $additionalHours = $db->loadObjectList();
        
        $result = array();
        
        if (!empty($additionalHours)) {
            $userTimezone = $this->getUsersOrSelectedTimezone();
            $currentDate = new DateTime('now', new DateTimeZone($userTimezone));
            // Set time to start of day for proper date comparison
            $currentDate->setTime(0, 0, 0);
            
            $showFuture = $this->showFutureHours;
            $numberToShow = $this->numberFutureHours;
            
            foreach ($additionalHours as $hour) {
                try {
                    // Create dates and handle timezone issues
                    $dateFromOriginal = new DateTime($hour->date_from);
                    $dateFromOriginal->setTime(12, 0, 0); // Set to noon to avoid timezone day shifts

                    $dateToOriginal = new DateTime($hour->date_to);
                    $dateToOriginal->setTime(12, 0, 0);
                                        
                    // Create separate dates for comparison (with time set for proper comparison)
                    $dateFromCompare = clone $dateFromOriginal;
                    $dateToCompare = clone $dateToOriginal;
                    $dateFromCompare->setTime(0, 0, 0);
                    $dateToCompare->setTime(23, 59, 59);
                    
                    // Ensure dateTo is not before dateFrom
                    if ($dateToCompare < $dateFromCompare) {
                        $dateToCompare = clone $dateFromCompare;
                        $dateToCompare->setTime(23, 59, 59);
                        $dateToOriginal = clone $dateFromOriginal;
                    }
                } catch (Exception $e) {
                    // Skip invalid dates
                    continue;
                }
                
                // If show_future is enabled, only show dates that haven't ended yet
                if ($showFuture && $dateToCompare < $currentDate) {
                    continue;
                }
                
                $result[] = array(
                    'date_from' => $dateFromOriginal,
                    'date_to' => $dateToOriginal,
                    'open_or_closed' => $hour->open_or_closed ?? 'closed',
                    'fullday' => $hour->fullday ?? '0',
                    'start_time' => $hour->start_time ?? '',
                    'end_time' => $hour->end_time ?? '',
                    'reason' => $hour->reason ?? ''
                );
            }
            
            // Sort by date
            usort($result, function($a, $b) {
                return $a['date_from'] <=> $b['date_from'];
            });
            
            // Limit to the number specified
            if ($showFuture && count($result) > $numberToShow) {
                $result = array_slice($result, 0, $numberToShow);
            }
        }
        
        return $result;
    }

    /**
     * Get HTML for additional opening hours
     *
     * @return  string
     */
    public function getAdditionalOpeningHoursHTML() 
    {
        $additionalHours = $this->getAdditionalOpeningHours();
        $htmlCode = '';
        
        if (!empty($additionalHours)) {
            $htmlCode .= '<div class="scs-additional-hours">';
            $htmlCode .= '<h4>' . Text::_('COM_SCORPIONOPENINGHOURS_ADDITIONAL_HOURS_LABEL') . '</h4>';
            
            foreach ($additionalHours as $hour) {
                $htmlCode .= '<div class="scs-additional-hour-item">';
                
                // Date range
                $dateFromFormatted = $hour['date_from']->format('d-m-Y');
                $dateToFormatted = $hour['date_to']->format('d-m-Y');
                
                if ($dateFromFormatted === $dateToFormatted) {
                    $htmlCode .= '<div class="scs-additional-date">' . $dateFromFormatted . '</div>';
                } else {
                    $htmlCode .= '<div class="scs-additional-date">' . $dateFromFormatted . ' - ' . $dateToFormatted . '</div>';
                }
                
                // Status and times
                if ($hour['open_or_closed'] === 'closed') {
                    $htmlCode .= '<div class="scs-additional-status scs-closed">' . ($this->openingHoursData->closed_text ?? 'CLOSED') . '</div>';
                } else {
                    $htmlCode .= '<div class="scs-additional-status scs-open">';
                    if ($hour['fullday'] === '1') {
                        $htmlCode .= ($this->openingHoursData->open_text ?? 'OPEN') . ' (' . Text::_('COM_SCORPIONOPENINGHOURS_FULLDAY_TEXT') . ')';
                    } else {
                        $htmlCode .= ($this->openingHoursData->open_text ?? 'OPEN') . ' (' . $hour['start_time'] . ' - ' . $hour['end_time'] . ')';
                    }
                    $htmlCode .= '</div>';
                }
                
                // Reason
                if (!empty($hour['reason'])) {
                    $htmlCode .= '<div class="scs-additional-reason">' . htmlspecialchars($hour['reason']) . '</div>';
                }
                
                $htmlCode .= '</div>';
            }
            
            $htmlCode .= '</div>';
        }
        
        return $htmlCode;
    }
}
